/**
 * Utilities for the admin dashboard
 */

jQuery( document ).ready( function( $ ) {
	'use strict';

	// Select Icon on Click.
	$( 'body' ).on( 'click', '.js-selectable-icon', function ( ev ) {
		ev.preventDefault();
		var $this = $( this );
		$this.siblings( '.js-icon-input' ).val( $this.data( 'iconname' ) ).change();
	} );

} );


/********************************************************
 			Backbone code for repeating fields in widgets
********************************************************/

// Namespace for Backbone elements.
window.ConsultPress = {
	Models:    {},
	ListViews: {},
	Views:     {},
	Utils:     {},
};

/**
 ******************** Backbone Models *******************
 */

_.extend( ConsultPress.Models, {
	Timetable: Backbone.Model.extend( {
		defaults: {
			'link':        '',
			'month':       '',
			'day':         '',
			'title':       '',
			'description': '',
			'info':        '',
		}
	} ),

	ContactProfileItem: Backbone.Model.extend( {
		defaults: {
			'text': '',
			'icon': 'fa-home',
		}
	} ),

	PricingPackageItem: Backbone.Model.extend( {
		defaults: {
			'text': '',
			'icon': 'fa-check',
		}
	} )
} );

/**
 ******************** Backbone Views *******************
 */

// Generic single view that others can extend from.
ConsultPress.Views.Abstract = Backbone.View.extend( {
	initialize: function ( params ) {
		this.templateHTML = params.templateHTML;

		return this;
	},

	render: function () {
		this.$el.html( Mustache.render( this.templateHTML, this.model.attributes ) );

		return this;
	},

	destroy: function ( ev ) {
		ev.preventDefault();

		this.remove();
		this.model.trigger( 'destroy' );
	},
} );

_.extend( ConsultPress.Views, {

	// View of a single timetable row.
	Timetable: ConsultPress.Views.Abstract.extend( {
		className: 'pt-widget-single-timetable',

		events: {
			'click .js-pt-remove-timetable-item': 'destroy',
		},
	} ),

	// View of a single contact profile item.
	ContactProfileItem: ConsultPress.Views.Abstract.extend( {
		className: 'pt-widget-single-contact-profile-item',

		events: {
			'click .js-pt-remove-contact-profile-item': 'destroy',
		},

		render: function () {
			this.$el.html( Mustache.render( this.templateHTML, this.model.attributes ) );
			this.$( 'input.js-icon-input' ).val( this.model.get( 'icon' ) );
			return this;
		},
	} ),

	// View of a single pricing package item.
	PricingPackageItem: ConsultPress.Views.Abstract.extend( {
		className: 'pt-widget-single-pricing-package-item',

		events: {
			'click .js-pt-remove-pricing-package-item': 'destroy',
		},

		render: function () {
			this.$el.html( Mustache.render( this.templateHTML, this.model.attributes ) );
			this.$( 'input.js-icon-input' ).val( this.model.get( 'icon' ) );
			return this;
		},
	} )
} );



/**
 ******************** Backbone ListViews *******************
 *
 * Parent container for multiple view nodes.
 */

ConsultPress.ListViews.Abstract = Backbone.View.extend( {

	initialize: function ( params ) {
		this.widgetId     = params.widgetId;
		this.itemsModel   = params.itemsModel;
		this.itemView     = params.itemView;
		this.itemTemplate = params.itemTemplate;

		// Cached reference to the element in the DOM.
		this.$items = this.$( params.itemsClass );

		// Collection of items(locations, people, testimonials,...).
		this.items = new Backbone.Collection( [], {
			model: this.itemsModel
		} );

		// Listen to adding of the new items.
		this.listenTo( this.items, 'add', this.appendOne );

		return this;
	},

	addNew: function ( ev ) {
		ev.preventDefault();

		var currentMaxId = this.getMaxId();

		this.items.add( new this.itemsModel( {
			id: (currentMaxId + 1)
		} ) );

		return this;
	},

	getMaxId: function () {
		if ( this.items.isEmpty() ) {
			return -1;
		}
		else {
			var itemWithMaxId = this.items.max( function ( item ) {
				return parseInt( item.id, 10 );
			} );

			return parseInt( itemWithMaxId.id, 10 );
		}
	},

	appendOne: function ( item ) {
		var renderedItem = new this.itemView( {
			model:        item,
			templateHTML: jQuery( this.itemTemplate + this.widgetId ).html()
		} ).render();

		var currentWidgetId = this.widgetId;

		// If the widget is in the initialize state (hidden), then do not append a new item.
		if ( '__i__' !== currentWidgetId.slice( -5 ) ) {
			this.$items.append( renderedItem.el );
		}

		return this;
	}
} );

_.extend( ConsultPress.ListViews, {

	// Collection of all timetable items, but associated with each individual widget.
	Timetables: ConsultPress.ListViews.Abstract.extend( {
		events: {
			'click .js-pt-add-timetable-item': 'addNew'
		}
	} ),

	// Collection of all contact profile items, but associated with each individual widget.
	ContactProfileItems: ConsultPress.ListViews.Abstract.extend( {
		events: {
			'click .js-pt-add-contact-profile-item': 'addNew'
		}
	} ),

	// Collection of all pricing package items, but associated with each individual widget.
	PricingPackageItems: ConsultPress.ListViews.Abstract.extend( {
		events: {
			'click .js-pt-add-pricing-package-item': 'addNew'
		}
	} )
} );

/**
 ******************** Repopulate Functions *******************
 */

_.extend( ConsultPress.Utils, {
	// Generic repopulation function used in all repopulate functions
	repopulateGeneric: function ( collectionType, parameters, json, widgetId ) {
		var collection = new collectionType( parameters );

		// Convert to array if needed
		if ( _( json ).isObject() ) {
			json = _( json ).values();
		}

		// Add all items to collection of newly created view
		collection.items.add( json, { parse: true } );
	},

	/**
	 * Function which adds the existing timetable items to the DOM
	 * @param  {json} timetableJSON
	 * @param  {string} widgetId ID of widget from PHP $this->id
	 * @return {void}
	 */
	repopulateTimetable: function ( timetableJSON, widgetId ) {
		var parameters = {
			el:           '#timetable-' + widgetId,
			widgetId:     widgetId,
			itemsClass:   '.timetable-items',
			itemTemplate: '#js-pt-timetable-',
			itemsModel:   ConsultPress.Models.Timetable,
			itemView:     ConsultPress.Views.Timetable,
		};

		this.repopulateGeneric( ConsultPress.ListViews.Timetables, parameters, timetableJSON, widgetId );
	},

	/**
	 * Function which adds the existing contact profile items to the DOM
	 * @param  {json} contactProfileItemJSON
	 * @param  {string} widgetId ID of widget from PHP $this->id
	 * @return {void}
	 */
	repopulateContactProfileItems: function ( contactProfileItemJSON, widgetId ) {
		var parameters = {
			el:           '#contact-profile-items-' + widgetId,
			widgetId:     widgetId,
			itemsClass:   '.contact-profile-items',
			itemTemplate: '#js-pt-contact-profile-item-',
			itemsModel:   ConsultPress.Models.ContactProfileItem,
			itemView:     ConsultPress.Views.ContactProfileItem,
		};

		this.repopulateGeneric( ConsultPress.ListViews.ContactProfileItems, parameters, contactProfileItemJSON, widgetId );
	},

	/**
	 * Function which adds the existing pricing package items to the DOM
	 * @param  {json} pricingPackageItemJSON
	 * @param  {string} widgetId ID of widget from PHP $this->id
	 * @return {void}
	 */
	repopulatePricingPackageItems: function ( pricingPackageItemJSON, widgetId ) {
		var parameters = {
			el:           '#pricing-package-items-' + widgetId,
			widgetId:     widgetId,
			itemsClass:   '.pricing-package-items',
			itemTemplate: '#js-pt-pricing-package-item-',
			itemsModel:   ConsultPress.Models.PricingPackageItem,
			itemView:     ConsultPress.Views.PricingPackageItem,
		};

		this.repopulateGeneric( ConsultPress.ListViews.PricingPackageItems, parameters, pricingPackageItemJSON, widgetId );
	}
} );
