<?php
/**
 * Customizer
 *
 * @package consultpress-pt
 */

use ProteusThemes\CustomizerUtils\Setting;
use ProteusThemes\CustomizerUtils\Control;
use ProteusThemes\CustomizerUtils\CacheManager;
use ProteusThemes\CustomizerUtils\Helpers as WpUtilsHelpers;

/**
 * Contains methods for customizing the theme customization screen.
 *
 * @link http://codex.wordpress.org/Theme_Customization_API
 */
class ConsultPress_Customizer_Base {
	/**
	 * The singleton manager instance
	 *
	 * @see wp-includes/class-wp-customize-manager.php
	 * @var WP_Customize_Manager
	 */
	protected $wp_customize;

	/**
	 * Instance of the DynamicCSS cache manager
	 *
	 * @var ProteusThemes\CustomizerUtils\CacheManager
	 */
	private $dynamic_css_cache_manager;

	/**
	 * Holds the array for the DynamiCSS.
	 *
	 * @var array
	 */
	private $dynamic_css = array();

	/**
	 * Constructor method for this class.
	 *
	 * @param WP_Customize_Manager $wp_customize The WP customizer manager instance.
	 */
	public function __construct( WP_Customize_Manager $wp_customize ) {
		// Set the private propery to instance of wp_customize.
		$this->wp_customize = $wp_customize;

		// Set the private propery to instance of DynamicCSS CacheManager.
		$this->dynamic_css_cache_manager = new CacheManager( $this->wp_customize );

		// Init the dynamic_css property.
		$this->dynamic_css = $this->dynamic_css_init();

		// Register the settings/panels/sections/controls.
		$this->register_settings();
		$this->register_panels();
		$this->register_sections();
		$this->register_partials();
		$this->register_controls();
		$this->register_kirki_fields();

		/**
		 * Action and filters
		 */

		// Render the CSS and cache it to the theme_mod when the setting is saved.
		add_action( 'wp_head', array( 'ProteusThemes\CustomizerUtils\Helpers', 'add_dynamic_css_style_tag' ), 50, 0 );
		add_action( 'customize_save_after', function() {
			$this->dynamic_css_cache_manager->cache_rendered_css();
		}, 10, 0 );

		// Save logo width/height dimensions.
		add_action( 'customize_save_logo_img', array( 'ProteusThemes\CustomizerUtils\Helpers', 'save_logo_dimensions' ), 10, 1 );
	}


	/**
	 * Initialization of the dynamic CSS settings with config arrays
	 *
	 * @return array
	 */
	private function dynamic_css_init() {
		$darken3   = new Setting\DynamicCSS\ModDarken( 3 );
		$darken5   = new Setting\DynamicCSS\ModDarken( 5 );
		$darken6   = new Setting\DynamicCSS\ModDarken( 6 );
		$darken12  = new Setting\DynamicCSS\ModDarken( 12 );
		$main_navigation_background_gradient = new Setting\DynamicCSS\ModLinearGradient( $darken5, $orientation = 'to right' );

		return array(
			'main_navigation_mobile_background' => array(
				'default' => '#273a42',
				'css_props' => array(
					array(
						'name' => 'background',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.main-navigation',
								'.sidebar__main-navigation .social-icons',
							),
						),
						'modifier'  => $main_navigation_background_gradient,
					),
				),
			),

			'main_navigation_mobile_color' => array(
				'default' => '#ffffff',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.main-navigation a',
							),
						),
					),
				),
			),

			'main_navigation_mobile_color_hover' => array(
				'default' => '#ffffff',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.main-navigation .menu-item:focus > a',
								'.main-navigation .menu-item:hover > a',
							),
						),
					),
				),
			),

			'main_navigation_mobile_sub_bgcolor' => array(
				'default' => '#47575e',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.main-navigation .sub-menu a',
							),
						),
					),
				),
			),

			'main_navigation_mobile_sub_color' => array(
				'default' => '#b3b8ba',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.main-navigation .sub-menu .menu-item > a',
							),
						),
					),
				),
			),

			'main_navigation_mobile_sub_color_hover' => array(
				'default' => '#eeeeee',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.main-navigation .sub-menu .menu-item:hover > a',
								'.main-navigation .sub-menu .menu-item:focus > a',
							),
						),
					),
				),
			),

			'main_navigation_background' => array(
				'default' => '#273a42',
				'css_props' => array(
					array(
						'name' => 'background',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.main-navigation',
								'.sidebar__main-navigation .social-icons',
							),
						),
						'modifier'  => $main_navigation_background_gradient,
					),
				),
			),

			'main_navigation_color' => array(
				'default' => '#eeeeee',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.main-navigation a',
							),
						),
					),
				),
			),

			'main_navigation_color_hover' => array(
				'default' => '#eeeeee',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.main-navigation > .menu-item:focus > a',
								'.main-navigation > .menu-item:hover > a',
								'.main-navigation > .current-menu-item > a',
								'.main-navigation > .current-menu-ancestor > a',
							),
						),
					),
				),
			),

			'main_navigation_sub_bg' => array(
				'default' => '#0bcda5',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.main-navigation .sub-menu a',
								'.main-navigation > .menu-item:focus::before',
								'.main-navigation > .menu-item:hover::before',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.main-navigation .sub-menu .menu-item:focus > a',
								'.main-navigation .sub-menu .menu-item:hover > a',
								'.main-navigation .sub-menu a::after',
							),
						),
						'modifier'  => $darken3,
					),
				),
			),

			'main_navigation_sub_color' => array(
				'default' => '#ffffff',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.main-navigation .sub-menu .menu-item a',
								'.main-navigation .sub-menu .menu-item:focus > a',
								'.main-navigation .sub-menu .menu-item:hover > a',
							),
						),
					),
				),
			),

			'page_header_bg_color' => array(
				'default' => '#1f2e34',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'@media (min-width: 992px)' => array(
								'.header__container',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'@media (max-width: 991px)' => array(
								'.header__page-header',
							),
						),
					),
				),
			),

			'page_header_color' => array(
				'default' => '#ffffff',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.page-header__title',
							),
						),
					),
				),
			),

			'page_header_subtitle_color' => array(
				'default' => '#eeeeee',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.page-header__subtitle',
							),
						),
					),
				),
			),

			'breadcrumbs_color' => array(
				'default'    => '#273a42',
				'css_props' => array(
					array(
						'name'      => 'color',
						'selectors' => array(
							'noop' => array(
								'.breadcrumbs a',
							),
						),
					),
				),
			),

			'breadcrumbs_color_hover' => array(
				'default'    => '#273a42',
				'css_props' => array(
					array(
						'name'      => 'color',
						'selectors' => array(
							'noop' => array(
								'.breadcrumbs a:focus',
								'.breadcrumbs a:hover',
								'.breadcrumbs a:active:hover',
							),
						),
					),
				),
			),

			'breadcrumbs_color_active' => array(
				'default'    => '#777777',
				'css_props' => array(
					array(
						'name'      => 'color',
						'selectors' => array(
							'noop' => array(
								'.breadcrumbs .current-item',
							),
						),
					),
				),
			),

			'text_color_content_area' => array(
				'default' => '#777777',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.content-area',
								'.article .icon-box__subtitle',
								'.timetable__description',
							),
						),
					),
				),
			),

			'headings_color' => array(
				'default' => '#273a42',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'h1',
								'h2',
								'h3',
								'h4',
								'h5',
								'h6',
								'.header__logo-text',
								'.timetable__title',
								'.pricing-package__price',
								'.article .icon-box__title',
								'.time-table .widget-title',
								'.time-table .week-day.today',
								'.brochure-box',
								'.latest-news__title a',
								'.latest-news__title a:focus',
								'.latest-news__title a:hover',
								'.latest-news__title a:active:hover',
								'.accordion__panel .panel-title a.collapsed',
								'.accordion__panel .panel-title a',
								'.testimonial__author',
								'.number-counter__number',
								'.number-counter__title',
								'.page-box__title a',
								'.page-box__title a:focus',
								'.page-box__title a:hover',
								'.page-box__title a:active:hover',
								'.article__title-link',
								'.article__title-link:focus',
								'.article__title-link:hover',
								'.article__title-link:active:hover',
								'.comment__author',
								'.comment__author a',
								'.comment__author a:focus',
								'.comment__author a:hover',
								'.comment__author a:active:hover',
							),
						),
					),
				),
			),

			'primary_color' => array(
				'default' => '#0bcda5',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.article__content .more-link:focus',
								'.widget_search .search-submit',
								'.header__logo:focus .header__logo-text',
								'.header__logo:hover .header__logo-text',
								'.sidebar__main-navigation .social-icons__link',
								'.pricing-package__details .fa',
								'.contact-profile__icon',
								'.open-position__details-item-icon',
								'.open-position__content .read-more',
								'.open-position__content .read-more:focus',
								'.article .icon-box .fa',
								'.accordion__panel .panel-title a::after',
								'.person-profile__social-icon',
								'.person-profile__social-icon:focus',
								'.accordion .more-link::after',
								'.accordion__panel .panel-title a:hover',
								'.brochure-box .fa',
								'.social-icons__link:focus',
								'.social-icons__link:hover',
							),
						),
					),
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.article__content .more-link:hover',
								'.sidebar__main-navigation .social-icons__link:focus',
								'.sidebar__main-navigation .social-icons__link:hover',
								'.open-position__content .read-more:hover',
								'.person-profile__social-icon:hover',
								'.social-icons__link:active:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.article__content .more-link:active:hover',
								'.open-position__content .read-more:active:hover',
								'.person-profile__social-icon:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.article__categories a',
								'.article__categories a:focus',
								'.btn-primary',
								'.btn-primary:focus',
								'.widget_calendar caption',
								'.timetable__date',
								'.open-position__details-location',
								'.article a.icon-box:focus',
								'.article a.icon-box:hover',
								'.person-profile__label',
								'.brochure-box:focus',
								'.brochure-box:hover',
								'.testimonial__quote::before',
								'.post-password-form input',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.article__categories a:hover',
								'.btn-primary:hover',
								'.brochure-box:active:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.article__categories a:active:hover',
								'.btn-primary:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-primary',
								'.btn-primary:focus',
								'.article a.icon-box:focus',
								'.article a.icon-box:hover',
								'.post-password-form input',
								'blockquote',
							),
						),
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-primary:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-primary:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'secondary_color' => array(
				'default' => '#e13c5f',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.pricing-package--featured .pricing-package__title',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-secondary',
								'.btn-secondary:focus',
								'.pricing-package--featured .pricing-package__label',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-secondary:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-secondary:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-secondary',
								'.btn-secondary:focus',
							),
						),
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-secondary:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-secondary:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'link_color' => array(
				'default' => '#0bcda5',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'a',
								'a:focus',
								'.open-position__content .read-more',
								'.open-position__content .read-more:focus',
								'.latest-news__more-news',
								'.latest-news__more-news:focus',
								'.latest-news--more-news',
								'.latest-news--more-news:focus',
								'.accordion .more-link',
								'.accordion .more-link:focus',
								'.page-box__more-link',
								'.page-box__more-link:focus',
								'.article__content .more-link',
								'.article__content .more-link:focus',
							),
						),
					),
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'a:hover',
								'.open-position__content .read-more:hover',
								'.latest-news__more-news:hover',
								'.latest-news--more-news:hover',
								'.accordion .more-link:hover',
								'.page-box__more-link:hover',
								'.article__content .more-link:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'a:active:hover',
								'.open-position__content .read-more:active:hover',
								'.latest-news__more-news:active:hover',
								'.latest-news--more-news:active:hover',
								'.accordion .more-link:active:hover',
								'.page-box__more-link:active:hover',
								'.article__content .more-link:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'dark_button_color' => array(
				'default' => '#273a42',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-dark',
								'.btn-dark:focus',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-dark:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-dark:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-dark',
								'.btn-dark:focus',
							),
						),
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-dark:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-dark:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'light_button_color' => array(
				'default' => '#888888',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-light',
								'.btn-light:focus',
							),
						),
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-light:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.btn-light:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-light',
								'.btn-light:focus',
							),
						),
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-light:hover',
							),
						),
						'modifier'  => $darken6,
					),
					array(
						'name' => 'border-color',
						'selectors' => array(
							'noop' => array(
								'.btn-light:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'body_bg' => array(
				'default'   => '#ffffff',
				'css_props' => array(
					array(
						'name'      => 'background-color',
						'selectors' => array(
							'noop' => array(
								'body .boxed-container',
							),
						),
					),
				),
			),

			'footer_bg_color' => array(
				'default' => '#273a42',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.footer__container',
							),
						),
					),
				),
			),

			'footer_title_color' => array(
				'default' => '#ffffff',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-top__heading',
							),
						),
					),
				),
			),

			'footer_text_color' => array(
				'default' => '#a9aeaf',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-top',
								'.footer__featured',
							),
						),
					),
				),
			),

			'footer_link_color' => array(
				'default' => '#a9aeaf',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-top a',
								'.footer-top .widget_nav_menu .menu a',
								'.footer .widget_recent_entries a',
								'.footer .widget_categories a',
								'.footer .widget_pages a',
								'.footer .widget_archive a',
								'.footer .widget_rss a',
								'.footer .widget_recent_comments a',
								'.footer .widget_meta a',
							),
						),
					),
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-top a:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'footer_bottom_text_color' => array(
				'default' => '#a9aeaf',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-bottom',
							),
						),
					),
				),
			),

			'footer_bottom_link_color' => array(
				'default' => '#0bcda5',
				'css_props' => array(
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-bottom a',
							),
						),
					),
					array(
						'name' => 'color',
						'selectors' => array(
							'noop' => array(
								'.footer-bottom a:active:hover',
							),
						),
						'modifier'  => $darken12,
					),
				),
			),

			'footer_featured_bg_color' => array(
				'default' => '#435157',
				'css_props' => array(
					array(
						'name' => 'background-color',
						'selectors' => array(
							'noop' => array(
								'.footer__featured',
							),
						),
					),
				),
			),
		);
	}

	/**
	 * Register customizer settings
	 *
	 * @return void
	 */
	public function register_settings() {
		// Branding.
		$this->wp_customize->add_setting( 'logo_img' );
		$this->wp_customize->add_setting( 'logo2x_img' );
		$this->wp_customize->add_setting( 'logo_top_margin', array( 'default' => 0 ) );

		// Featured page.
		$this->wp_customize->add_setting( 'featured_page_select', array( 'default' => 'none' ) );
		$this->wp_customize->add_setting( 'featured_page_custom_text' );
		$this->wp_customize->add_setting( 'featured_page_custom_url' );
		$this->wp_customize->add_setting( 'featured_page_open_in_new_window' );

		// Navigation.
		$this->wp_customize->add_setting( 'navigation_social_icons_intro_text', array( 'default' => 'Get social with us:' ) );

		// Page header area.
		$this->wp_customize->add_setting( 'page_header_bg_img' );
		$this->wp_customize->add_setting( 'page_header_bg_img_repeat', array( 'default' => 'repeat' ) );
		$this->wp_customize->add_setting( 'page_header_bg_img_position_x', array( 'default' => 'left' ) );
		$this->wp_customize->add_setting( 'page_header_bg_img_attachment', array( 'default' => 'scroll' ) );

		// Typography.
		$this->wp_customize->add_setting( 'charset_setting', array( 'default' => 'latin' ) );

		// Theme layout & color.
		$this->wp_customize->add_setting( 'layout_mode', array( 'default' => 'wide' ) );
		$this->wp_customize->add_setting( 'body_bg_img' );
		$this->wp_customize->add_setting( 'body_bg_img_repeat', array( 'default' => 'repeat' ) );
		$this->wp_customize->add_setting( 'body_bg_img_position_x', array( 'default' => 'left' ) );
		$this->wp_customize->add_setting( 'body_bg_img_attachment', array( 'default' => 'scroll' ) );

		// Footer.
		$this->wp_customize->add_setting( 'footer_featured_text', array( 'default' => '<img class="img-fluid" src="https://demo.proteusthemes.com/consultpress/wp-content/uploads/sites/44/2016/12/logo-light.png" alt="Logo Footer"><hr class="hr-light"><p>Energistically drive sustainable scas for high-payoff testing procedures and business plans.</p>[button style="secondary" fullwidth="true"]Sign Up[/button]' ) );

		$this->wp_customize->add_setting( 'footer_widgets_layout', array( 'default' => '[4,8]' ) );

		$this->wp_customize->add_setting( 'footer_bottom_txt', array( 'default' => '&copy; ' . date( 'Y' ) . ' <strong><a href="https://www.proteusthemes.com/wordpress-themes/consultpress/">ConsultPress</a></strong> All Rights Reserved.' ) );

		// Custom code (css/js).
		$this->wp_customize->add_setting( 'custom_js_head' );
		$this->wp_customize->add_setting( 'custom_js_footer' );

		// ACF.
		$this->wp_customize->add_setting( 'show_acf', array( 'default' => 'no' ) );
		$this->wp_customize->add_setting( 'use_minified_css', array( 'default' => 'no' ) );

		// All the DynamicCSS settings.
		foreach ( $this->dynamic_css as $setting_id => $args ) {
			$this->wp_customize->add_setting(
				new Setting\DynamicCSS( $this->wp_customize, $setting_id, $args )
			);
		}
	}


	/**
	 * Panels
	 *
	 * @return void
	 */
	public function register_panels() {
		// One ProteusThemes panel to rule them all.
		$this->wp_customize->add_panel( 'panel_consultpress', array(
			'title'       => esc_html__( '[PT] Theme Options', 'consultpress-pt' ),
			'description' => esc_html__( 'All ConsultPress theme specific settings.', 'consultpress-pt' ),
			'priority'    => 10,
		) );
	}


	/**
	 * Sections
	 *
	 * @return void
	 */
	public function register_sections() {
		$this->wp_customize->add_section( 'consultpress_section_logos', array(
			'title'       => esc_html__( 'Logo', 'consultpress-pt' ),
			'description' => sprintf( esc_html__( 'Logo for the ConsultPress theme.', 'consultpress-pt' ) , '<i>', '</i>' ),
			'priority'    => 10,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'consultpress_section_navigation', array(
			'title'       => esc_html__( 'Navigation', 'consultpress-pt' ),
			'description' => esc_html__( 'Navigation for the ConsultPress theme.', 'consultpress-pt' ),
			'priority'    => 30,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'consultpress_section_page_header', array(
			'title'       => esc_html__( 'Page Header Area', 'consultpress-pt' ),
			'description' => esc_html__( 'All layout and appearance settings for the page header area (regular pages).', 'consultpress-pt' ),
			'priority'    => 33,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'consultpress_section_breadcrumbs', array(
			'title'       => esc_html__( 'Breadcrumbs', 'consultpress-pt' ),
			'description' => esc_html__( 'All layout and appearance settings for breadcrumbs.', 'consultpress-pt' ),
			'priority'    => 35,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'consultpress_section_theme_colors', array(
			'title'       => esc_html__( 'Theme Layout &amp; Colors', 'consultpress-pt' ),
			'priority'    => 40,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'section_footer', array(
			'title'       => esc_html__( 'Footer', 'consultpress-pt' ),
			'description' => esc_html__( 'All layout and appearance settings for the footer.', 'consultpress-pt' ),
			'priority'    => 90,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'section_custom_code', array(
			'title'       => esc_html__( 'Custom Code' , 'consultpress-pt' ),
			'priority'    => 100,
			'panel'       => 'panel_consultpress',
		) );

		$this->wp_customize->add_section( 'section_other', array(
			'title'       => esc_html__( 'Other' , 'consultpress-pt' ),
			'priority'    => 150,
			'panel'       => 'panel_consultpress',
		) );
	}


	/**
	 * Partials for selective refresh
	 *
	 * @return void
	 */
	public function register_partials() {
		$this->wp_customize->selective_refresh->add_partial( 'dynamic_css', array(
			'selector' => 'head > #wp-utils-dynamic-css-style-tag',
			'settings' => array_keys( $this->dynamic_css ),
			'render_callback' => function() {
				return $this->dynamic_css_cache_manager->render_css();
			},
		) );
	}


	/**
	 * Controls
	 *
	 * @return void
	 */
	public function register_controls() {
		// Section: consultpress_section_logos.
		$this->wp_customize->add_control( new WP_Customize_Image_Control(
			$this->wp_customize,
			'logo_img',
			array(
				'label'       => esc_html__( 'Logo Image', 'consultpress-pt' ),
				'description' => esc_html__( 'Max recommended height for the logo image is 80px.', 'consultpress-pt' ),
				'section'     => 'consultpress_section_logos',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Image_Control(
			$this->wp_customize,
			'logo2x_img',
			array(
				'label'       => esc_html__( 'Retina Logo Image', 'consultpress-pt' ),
				'description' => esc_html__( '2x logo size, for screens with high DPI.', 'consultpress-pt' ),
				'section'     => 'consultpress_section_logos',
			)
		) );
		$this->wp_customize->add_control(
			'logo_top_margin',
			array(
				'type'        => 'number',
				'label'       => esc_html__( 'Logo top margin', 'consultpress-pt' ),
				'description' => esc_html__( 'In pixels.', 'consultpress-pt' ),
				'section'     => 'consultpress_section_logos',
				'input_attrs' => array(
					'min'  => 0,
					'max'  => 120,
					'step' => 5,
				),
			)
		);

		// Section: consultpress_section_navigation.
		$this->wp_customize->add_control( 'featured_page_select', array(
			'type'        => 'select',
			'priority'    => 113,
			'label'       => esc_html__( 'Featured page', 'consultpress-pt' ),
			'description' => esc_html__( 'To which page should the Featured Page button link to?', 'consultpress-pt' ),
			'section'     => 'consultpress_section_navigation',
			'choices'     => WpUtilsHelpers::get_all_pages_id_title(),
		) );
		$this->wp_customize->add_control( 'featured_page_custom_text', array(
			'priority'        => 115,
			'label'           => esc_html__( 'Custom Button Text', 'consultpress-pt' ),
			'section'         => 'consultpress_section_navigation',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_specific_value( 'featured_page_select', 'custom-url' );
			},
		) );

		$this->wp_customize->add_control( 'featured_page_custom_url', array(
			'priority'        => 117,
			'label'           => esc_html__( 'Custom URL', 'consultpress-pt' ),
			'section'         => 'consultpress_section_navigation',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_specific_value( 'featured_page_select', 'custom-url' );
			},
		) );

		$this->wp_customize->add_control( 'featured_page_open_in_new_window', array(
			'type'            => 'checkbox',
			'priority'        => 120,
			'label'           => esc_html__( 'Open link in a new window/tab.', 'consultpress-pt' ),
			'section'         => 'consultpress_section_navigation',
			'active_callback' => function() {
				return ( ! WpUtilsHelpers::is_theme_mod_specific_value( 'featured_page_select', 'none' ) );
			},
		) );

		$this->wp_customize->add_control( 'navigation_social_icons_intro_text', array(
			'priority'        => 121,
			'label'           => esc_html__( 'Social icons intro text', 'consultpress-pt' ),
			'section'         => 'consultpress_section_navigation',
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_background',
			array(
				'priority' => 125,
				'label'    => esc_html__( 'Main navigation background color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_color',
			array(
				'priority' => 130,
				'label'    => esc_html__( 'Main navigation link color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_color_hover',
			array(
				'priority' => 132,
				'label'    => esc_html__( 'Main navigation link hover color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_sub_bg',
			array(
				'priority' => 160,
				'label'    => esc_html__( 'Main navigation submenu background', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_sub_color',
			array(
				'priority' => 170,
				'label'    => esc_html__( 'Main navigation submenu link color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_mobile_color',
			array(
				'priority' => 190,
				'label'    => esc_html__( 'Main navigation link color (mobile)', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_mobile_color_hover',
			array(
				'priority' => 192,
				'label'    => esc_html__( 'Main navigation link hover color (mobile)', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_mobile_sub_bgcolor',
			array(
				'priority' => 193,
				'label'    => esc_html__( 'Main navigation submenu background color (mobile)', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_mobile_sub_color',
			array(
				'priority' => 194,
				'label'    => esc_html__( 'Main navigation submenu link color (mobile)', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_mobile_sub_color_hover',
			array(
				'priority' => 195,
				'label'    => esc_html__( 'Main navigation submenu link hover color (mobile)', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'main_navigation_mobile_background',
			array(
				'priority' => 188,
				'label'    => esc_html__( 'Main navigation background color (mobile)', 'consultpress-pt' ),
				'section'  => 'consultpress_section_navigation',
			)
		) );

		// Section: consultpress_section_page_header.
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'page_header_bg_color',
			array(
				'priority' => 10,
				'label'    => esc_html__( 'Page Header background color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_page_header',
			)
		) );

		$this->wp_customize->add_control( new WP_Customize_Image_Control(
			$this->wp_customize,
			'page_header_bg_img',
			array(
				'priority' => 20,
				'label'    => esc_html__( 'Page Header Background Image', 'consultpress-pt' ),
				'section'  => 'consultpress_section_page_header',
			)
		) );
		$this->wp_customize->add_control( 'page_header_bg_img_repeat', array(
			'priority'        => 21,
			'label'           => esc_html__( 'Page Header Background Repeat', 'consultpress-pt' ),
			'section'         => 'consultpress_section_page_header',
			'type'            => 'radio',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_not_empty( 'page_header_bg_img' );
			},
			'choices'         => array(
				'no-repeat'  => esc_html__( 'No Repeat', 'consultpress-pt' ),
				'repeat'     => esc_html__( 'Tile', 'consultpress-pt' ),
				'repeat-x'   => esc_html__( 'Tile Horizontally', 'consultpress-pt' ),
				'repeat-y'   => esc_html__( 'Tile Vertically', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( 'page_header_bg_img_position_x', array(
			'priority'        => 22,
			'label'           => esc_html__( 'Page Header Background Position', 'consultpress-pt' ),
			'section'         => 'consultpress_section_page_header',
			'type'            => 'radio',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_not_empty( 'page_header_bg_img' );
			},
			'choices'         => array(
				'left'       => esc_html__( 'Left', 'consultpress-pt' ),
				'center'     => esc_html__( 'Center', 'consultpress-pt' ),
				'right'      => esc_html__( 'Right', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( 'page_header_bg_img_attachment', array(
			'priority'        => 23,
			'label'           => esc_html__( 'Page Header Background Attachment', 'consultpress-pt' ),
			'section'         => 'consultpress_section_page_header',
			'type'            => 'radio',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_not_empty( 'page_header_bg_img' );
			},
			'choices'         => array(
				'scroll'     => esc_html__( 'Scroll', 'consultpress-pt' ),
				'fixed'      => esc_html__( 'Fixed', 'consultpress-pt' ),
			),
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'page_header_color',
			array(
				'priority' => 30,
				'label'    => esc_html__( 'Page Header title color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_page_header',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'page_header_subtitle_color',
			array(
				'priority' => 31,
				'label'    => esc_html__( 'Page Header subtitle color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_page_header',
			)
		) );

		// Section: consultpress_section_breadcrumbs.
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'breadcrumbs_color',
			array(
				'priority' => 45,
				'label'    => esc_html__( 'Breadcrumbs text color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_breadcrumbs',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'breadcrumbs_color_hover',
			array(
				'priority' => 50,
				'label'    => esc_html__( 'Breadcrumbs hover text color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_breadcrumbs',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'breadcrumbs_color_active',
			array(
				'priority' => 50,
				'label'    => esc_html__( 'Breadcrumbs active text color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_breadcrumbs',
			)
		) );

		// Section: consultpress_section_theme_colors.
		$this->wp_customize->add_control( 'layout_mode', array(
			'type'     => 'select',
			'priority' => 10,
			'label'    => esc_html__( 'Layout', 'consultpress-pt' ),
			'section'  => 'consultpress_section_theme_colors',
			'choices'  => array(
				'wide'  => esc_html__( 'Wide', 'consultpress-pt' ),
				'boxed' => esc_html__( 'Boxed', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'text_color_content_area',
			array(
				'priority' => 30,
				'label'    => esc_html__( 'Text color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'headings_color',
			array(
				'priority' => 33,
				'label'    => esc_html__( 'Headings color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'primary_color',
			array(
				'priority' => 34,
				'label'    => esc_html__( 'Primary color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'secondary_color',
			array(
				'priority' => 35,
				'label'    => esc_html__( 'Secondary color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'link_color',
			array(
				'priority' => 36,
				'label'    => esc_html__( 'Link color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'dark_button_color',
			array(
				'priority' => 37,
				'label'    => esc_html__( 'Dark button background color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'light_button_color',
			array(
				'priority' => 38,
				'label'    => esc_html__( 'Light button background color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Image_Control(
			$this->wp_customize,
			'body_bg_img',
			array(
				'priority' => 40,
				'label'    => esc_html__( 'Body background image', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );
		$this->wp_customize->add_control( 'body_bg_img_repeat', array(
			'priority'        => 41,
			'label'           => esc_html__( 'Body background repeat', 'consultpress-pt' ),
			'section'         => 'consultpress_section_theme_colors',
			'type'            => 'radio',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_not_empty( 'body_bg_img' );
			},
			'choices'         => array(
				'no-repeat' => esc_html__( 'No Repeat', 'consultpress-pt' ),
				'repeat'    => esc_html__( 'Tile', 'consultpress-pt' ),
				'repeat-x'  => esc_html__( 'Tile Horizontally', 'consultpress-pt' ),
				'repeat-y'  => esc_html__( 'Tile Vertically', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( 'body_bg_img_position_x', array(
			'priority'        => 42,
			'label'           => esc_html__( 'Body background position', 'consultpress-pt' ),
			'section'         => 'consultpress_section_theme_colors',
			'type'            => 'radio',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_not_empty( 'body_bg_img' );
			},
			'choices'         => array(
				'left'   => esc_html__( 'Left', 'consultpress-pt' ),
				'center' => esc_html__( 'Center', 'consultpress-pt' ),
				'right'  => esc_html__( 'Right', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( 'body_bg_img_attachment', array(
			'priority'        => 43,
			'label'           => esc_html__( 'Body background attachment', 'consultpress-pt' ),
			'section'         => 'consultpress_section_theme_colors',
			'type'            => 'radio',
			'active_callback' => function() {
				return WpUtilsHelpers::is_theme_mod_not_empty( 'body_bg_img' );
			},
			'choices'         => array(
				'scroll' => esc_html__( 'Scroll', 'consultpress-pt' ),
				'fixed'  => esc_html__( 'Fixed', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'body_bg',
			array(
				'priority' => 45,
				'label'    => esc_html__( 'Body background color', 'consultpress-pt' ),
				'section'  => 'consultpress_section_theme_colors',
			)
		) );

		// Section: section_footer.
		$this->wp_customize->add_control( 'footer_featured_text', array(
			'priority'    => 4,
			'type'        => 'textarea',
			'label'       => esc_html__( 'Footer featured text', 'consultpress-pt' ),
			'description' => esc_html__( 'Custom text before the widgets area.', 'consultpress-pt' ),
			'section'     => 'section_footer',
		) );
		$this->wp_customize->add_control( new Control\LayoutBuilder(
			$this->wp_customize,
			'footer_widgets_layout',
			array(
				'priority'    => 5,
				'label'       => esc_html__( 'Footer widgets layout', 'consultpress-pt' ),
				'description' => esc_html__( 'Select number of widget you want in the footer and then with the slider rearrange the layout', 'consultpress-pt' ),
				'section'     => 'section_footer',
				'input_attrs' => array(
					'min'     => 0,
					'max'     => 12,
					'step'    => 1,
					'maxCols' => 6,
				),
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_bg_color',
			array(
				'priority' => 10,
				'label'    => esc_html__( 'Footer background color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );
		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_title_color',
			array(
				'priority' => 30,
				'label'    => esc_html__( 'Footer widget title color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_text_color',
			array(
				'priority' => 31,
				'label'    => esc_html__( 'Footer text color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_link_color',
			array(
				'priority' => 32,
				'label'    => esc_html__( 'Footer link color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_bottom_text_color',
			array(
				'priority' => 36,
				'label'    => esc_html__( 'Footer bottom text color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_bottom_link_color',
			array(
				'priority' => 37,
				'label'    => esc_html__( 'Footer bottom link color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );

		$this->wp_customize->add_control( new WP_Customize_Color_Control(
			$this->wp_customize,
			'footer_featured_bg_color',
			array(
				'priority' => 40,
				'label'    => esc_html__( 'Featured Footer background color', 'consultpress-pt' ),
				'section'  => 'section_footer',
			)
		) );

		$this->wp_customize->add_control( 'footer_bottom_txt', array(
			'type'        => 'text',
			'priority'    => 110,
			'label'       => esc_html__( 'Footer bottom text', 'consultpress-pt' ),
			'description' => esc_html__( 'You can use HTML: a, span, i, em, strong, img.', 'consultpress-pt' ),
			'section'     => 'section_footer',
		) );

		// Section: section_custom_code.
		$this->wp_customize->add_control( 'custom_js_head', array(
			'type'        => 'textarea',
			'label'       => esc_html__( 'Custom JavaScript (head)', 'consultpress-pt' ),
			'description' => esc_html__( 'You have to include the &lt;script&gt;&lt;/script&gt; tags as well. Paste your Google Analytics tracking code here.', 'consultpress-pt' ),
			'section'     => 'section_custom_code',
		) );

		$this->wp_customize->add_control( 'custom_js_footer', array(
			'type'        => 'textarea',
			'label'       => esc_html__( 'Custom JavaScript (footer)', 'consultpress-pt' ),
			'description' => esc_html__( 'You have to include the &lt;script&gt;&lt;/script&gt; tags as well.', 'consultpress-pt' ),
			'section'     => 'section_custom_code',
		) );

		// Section: section_other.
		$this->wp_customize->add_control( 'show_acf', array(
			'type'        => 'select',
			'label'       => esc_html__( 'Show ACF admin panel?', 'consultpress-pt' ),
			'description' => esc_html__( 'If you want to use ACF and need the ACF admin panel set this to <strong>Yes</strong>. Do not change if you do not know what you are doing.', 'consultpress-pt' ),
			'section'     => 'section_other',
			'choices'     => array(
				'no'  => esc_html__( 'No', 'consultpress-pt' ),
				'yes' => esc_html__( 'Yes', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( 'use_minified_css', array(
			'type'    => 'select',
			'label'   => esc_html__( 'Use minified theme CSS', 'consultpress-pt' ),
			'section' => 'section_other',
			'choices' => array(
				'no'  => esc_html__( 'No', 'consultpress-pt' ),
				'yes' => esc_html__( 'Yes', 'consultpress-pt' ),
			),
		) );
		$this->wp_customize->add_control( 'charset_setting', array(
			'type'     => 'select',
			'label'    => esc_html__( 'Character set for Google Fonts', 'consultpress-pt' ),
			'section'  => 'section_other',
			'choices'  => array(
				'latin'        => 'Latin',
				'latin-ext'    => 'Latin Extended',
				'cyrillic'     => 'Cyrillic',
				'cyrillic-ext' => 'Cyrillic Extended',
				'vietnamese'   => 'Vietnamese',
				'greek'        => 'Greek',
				'greek-ext'    => 'Greek Extended',
			),
		) );
	}

	/**
	 * Register Kirki customizer fields (controls and settings).
	 */
	public function register_kirki_fields() {
		// Social icons repeater field.
		Kirki::add_field( 'navigation_social_icons', array(
			'type'        => 'repeater',
			'label'       => esc_attr__( 'Social Icons', 'consultpress-pt' ),
			'description' => sprintf( esc_attr__( 'You can select the icons from the %1$sFontAwesome icon collection%2$s.', 'consultpress-pt' ) , '<a href="http://fontawesome.io/icons/" target="_blank">', '</a>' ),
			'section'     => 'consultpress_section_navigation',
			'priority'    => 122,
			'settings'    => 'navigation_social_icons',
			'row_label'   => array(
				'type'  => 'text',
				'value' => esc_attr__( 'social icon', 'consultpress-pt' ),
			),
			'default' => array(
				array(
					'icon' => 'fa-linkedin',
					'url'  => 'https://www.linkedin.com',
				),
				array(
					'icon' => 'fa-facebook',
					'url'  => 'https://www.facebook.com/ProteusThemes',
				),
				array(
					'icon' => 'fa-twitter',
					'url'  => 'https://twitter.com/ProteusThemes',
				),
			),
			'fields' => array(
				'icon' => array(
					'type'        => 'text',
					'label'       => esc_attr__( 'Font Awesome icon class', 'consultpress-pt' ),
					'description' => esc_attr__( 'Input a FontAwesome icon class, for example: fa-facebook', 'consultpress-pt' ),
					'default'     => 'fa-facebook',
				),
				'url' => array(
					'type'        => 'text',
					'label'       => esc_attr__( 'URL', 'consultpress-pt' ),
					'description' => esc_attr__( 'URL to your social network profile', 'consultpress-pt' ),
					'default'     => '',
				),
			),
		) );
	}
}
