<?php
/**
 * Contact Profile Widget
 *
 * @package consultpress-pt
 */

if ( ! class_exists( 'PW_Contact_Profile' ) ) {
	class PW_Contact_Profile extends WP_Widget {

		private $current_widget_id;
		private $font_awesome_icons_list;

		// Basic widget settings.
		function widget_id_base() { return 'contact_profile'; }
		function widget_name() { return esc_html__( 'Contact Profile', 'consultpress-pt' ); }
		function widget_description() { return esc_html__( 'Widget displaying contact info.', 'consultpress-pt' ); }
		function widget_class() { return 'widget-contact-profile'; }

		/**
		 * Register widget with WordPress.
		 */
		public function __construct() {
			parent::__construct(
				'pw_' . $this->widget_id_base(),
				sprintf( 'ProteusThemes: %s', $this->widget_name() ),
				array(
					'description' => $this->widget_description(),
					'classname'   => $this->widget_class(),
				)
			);
			// A list of icons to choose from in the widget backend.
			$this->font_awesome_icons_list = apply_filters(
				'pw/contact_profile_fa_icons_list',
				array(
					'fa-home',
					'fa-phone',
					'fa-envelope-o',
					'fa-envelope',
					'fa-map-marker',
					'fa-users',
					'fa-female',
					'fa-male',
					'fa-inbox',
					'fa-compass',
					'fa-laptop',
					'fa-money',
					'fa-suitcase',
					'fa-warning',
				)
			);
		}

		/**
		 * Front-end display of widget.
		 *
		 * @see WP_Widget::widget()
		 *
		 * @param array $args
		 * @param array $instance
		 */
		public function widget( $args, $instance ) {
			$items               = isset( $instance['items'] ) ? $instance['items'] : array();

			echo $args['before_widget'];
			?>
				<div class="card  contact-profile">
					<div class="card-block  contact-profile__container">
						<?php if ( ! empty( $instance['name'] ) ) : ?>
							<h4 class="contact-profile__name"><?php echo esc_html( $instance['name'] ); ?></h4>
						<?php endif; ?>
						<?php if ( ! empty( $items ) ) : ?>
							<div class="contact-profile__items">
								<?php foreach ( $items as $item ) : ?>
									<div class="contact-profile__item">
										<div class="contact-profile__icon">
											<i class="fa  <?php echo esc_attr( $item['icon'] ); ?>"></i>
										</div>
										<p class="contact-profile__text">
											<?php echo wp_kses_post( $item['text'] ); ?>
										</p>
									</div>
								<?php endforeach; ?>
							</div>
						<?php endif; ?>
					</div>
				</div>
			<?php
			echo $args['after_widget'];
		}

		/**
		 * Sanitize widget form values as they are saved.
		 *
		 * @param array $new_instance The new options.
		 * @param array $old_instance The previous options.
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = array();

			$instance['name'] = sanitize_text_field( $new_instance['name'] );

			if ( ! empty( $new_instance['items'] )  ) {
				foreach ( $new_instance['items'] as $key => $item ) {
					$instance['items'][ $key ]['id']   = sanitize_key( $item['id'] );
					$instance['items'][ $key ]['icon'] = sanitize_html_class( $item['icon'] );
					$instance['items'][ $key ]['text'] = wp_kses_post( $item['text'] );
				}
			}

			return $instance;
		}

		/**
		 * Back-end widget form.
		 *
		 * @param array $instance The widget options.
		 */
		public function form( $instance ) {

			$name  = empty( $instance['name'] ) ? '' : $instance['name'];
			$items = isset( $instance['items'] ) ? $instance['items'] : array();

			// Page Builder fix when using repeating fields.
			if ( 'temp' === $this->id ) {
				$this->current_widget_id = $this->number;
			}
			else {
				$this->current_widget_id = $this->id;
			}

			?>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'name' ) ); ?>"><?php esc_html_e( 'Name:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'name' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'name' ) ); ?>" type="text" value="<?php echo esc_attr( $name ); ?>" />
			</p>

			<hr>

			<h3><?php esc_html_e( 'Items:', 'consultpress-pt' ); ?></h3>

			<script type="text/template" id="js-pt-contact-profile-item-<?php echo esc_attr( $this->current_widget_id ); ?>">
				<p>
					<label for="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-text"><?php esc_html_e( 'Text:', 'consultpress-pt' ); ?></label>
					<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-text" name="<?php echo esc_attr( $this->get_field_name( 'items' ) ); ?>[{{id}}][text]" type="text" value="{{text}}" />
				</p>

				<p>
					<label for="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-icon"><?php esc_html_e( 'Select icon:', 'consultpress-pt' ); ?></label> <br />
					<small><?php printf( esc_html__( 'Click on the icon below or manually select from the %s website', 'consultpress-pt' ), '<a href="http://fontawesome.io/icons/" target="_blank">FontAwesome</a>' ); ?>.</small>
					<input id="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-icon" name="<?php echo esc_attr( $this->get_field_name( 'items' ) ); ?>[{{id}}][icon]" type="text" value="{{icon}}" class="widefat  js-icon-input" /> <br><br>
					<?php foreach ( $this->font_awesome_icons_list as $icon ) : ?>
						<a class="js-selectable-icon  icon-widget" href="#" data-iconname="<?php echo esc_attr( $icon ); ?>"><i class="fa fa-lg <?php echo esc_attr( $icon ); ?>"></i></a>
					<?php endforeach; ?>
				</p>

				<p>
					<input name="<?php echo esc_attr( $this->get_field_name( 'items' ) ); ?>[{{id}}][id]" type="hidden" value="{{id}}" />
					<a href="#" class="pt-remove-contact-profile-item  js-pt-remove-contact-profile-item"><span class="dashicons dashicons-dismiss"></span> <?php esc_html_e( 'Remove item', 'consultpress-pt' ); ?></a>
				</p>
			</script>
			<div class="pt-widget-contact-profile-items" id="contact-profile-items-<?php echo esc_attr( $this->current_widget_id ); ?>">
				<div class="contact-profile-items"></div>
				<p>
					<a href="#" class="button  js-pt-add-contact-profile-item"><?php esc_html_e( 'Add New Item', 'consultpress-pt' ); ?></a>
				</p>
			</div>

			<script type="text/javascript">
				(function() {
					// repopulate the form - items
					var contactProfileItemJSON = <?php echo wp_json_encode( $items ) ?>;

					// get the right widget id and remove the added < > characters at the start and at the end.
					var widgetId = '<<?php echo esc_js( $this->current_widget_id ); ?>>'.slice( 1, -1 );

					if ( _.isFunction( ConsultPress.Utils.repopulateContactProfileItems ) ) {
						ConsultPress.Utils.repopulateContactProfileItems( contactProfileItemJSON, widgetId );
					}
				})();
			</script>

			<?php
		}
	}
	register_widget( 'PW_Contact_Profile' );
}
