<?php
/**
 * Pricing Package Widget
 *
 * @package consultpress-pt
 */

if ( ! class_exists( 'PW_Pricing_Package' ) ) {
	class PW_Pricing_Package extends WP_Widget {
		private $widget_id_base, $widget_class, $widget_name, $widget_description;
		private $font_awesome_icons_list, $current_widget_id;

		public function __construct() {
			// Basic widget settings.
			$this->widget_id_base     = 'pricing_package';
			$this->widget_class       = 'widget-pricing-package';
			$this->widget_name        = esc_html__( 'Pricing Package', 'consultpress-pt' );
			$this->widget_description = esc_html__( 'Pricing package widget for Page Builder.', 'consultpress-pt' );

			parent::__construct(
				'pw_' . $this->widget_id_base,
				sprintf( 'ProteusThemes: %s', $this->widget_name ),
				array(
					'description' => $this->widget_description,
					'classname'   => $this->widget_class,
				)
			);

			// A list of icons to choose from in the widget backend.
			$this->font_awesome_icons_list = apply_filters(
				'pw/pricing_package_fa_icons_list',
				array(
					'fa-check',
					'fa-times',
					'fa-envelope',
					'fa-wrench',
					'fa-reply',
					'fa-laptop',
					'fa-gamepad',
					'fa-television',
					'fa-music',
					'fa-battery-full',
					'fa-ellipsis-v',
					'fa-apple',
					'fa-linux',
					'fa-windows',
					'fa-android',
					'fa-cogs',
					'fa-plug',
					'fa-volume-up',
				)
			);
		}

		/**
		 * Front-end display of widget.
		 *
		 * @see WP_Widget::widget()
		 *
		 * @param array $args
		 * @param array $instance
		 */
		public function widget( $args, $instance ) {
			$items       = isset( $instance['items'] ) ? array_values( $instance['items'] ) : array();
			$is_featured = ! empty( $instance['is_featured'] );
			$target      = ! empty( $instance['new_tab'] ) ? '_blank' : '_self';

			echo $args['before_widget'];
			?>
				<div class="pricing-package<?php echo $is_featured ? '  pricing-package--featured' : ''; ?>">
					<?php if ( ! empty( $instance['label'] ) ) : ?>
						<div class="pricing-package__label">
							<?php echo esc_html( $instance['label'] ); ?>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $instance['title'] ) ) : ?>
						<div class="h3  pricing-package__title">
							<?php echo wp_kses_post( $instance['title'] ); ?>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $instance['subtitle'] ) ) : ?>
						<div class="pricing-package__subtitle">
							<?php echo esc_html( $instance['subtitle'] ); ?>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $instance['price'] ) ) : ?>
						<div class="h4  pricing-package__price">
							<?php echo esc_html( $instance['price'] ); ?>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $items ) ) : ?>
						<ul class="pricing-package__details">
							<?php foreach ( $items as $item ) : ?>
								<li class="pricing-package__detail">
									<i class="fa  <?php echo esc_attr( $item['icon'] ); ?>" aria-hidden="true"></i> <?php echo wp_kses_post( $item['text'] ); ?>
								</li>
							<?php endforeach; ?>
						</ul>
					<?php endif; ?>

					<?php if ( ! empty( $instance['cta_text'] ) && ! empty( $instance['cta_url'] ) ) : ?>
						<a href="<?php echo esc_url( $instance['cta_url'] ); ?>" class="btn  <?php echo $is_featured ? 'btn-secondary' : 'btn-light'; ?>  btn-block  pricing-package__cta" target="<?php echo esc_attr( $target ); ?>">
							<?php echo esc_html( $instance['cta_text'] ); ?>
						</a>
					<?php endif; ?>
				</div>
			<?php
			echo $args['after_widget'];
		}

		/**
		 * Sanitize widget form values as they are saved.
		 *
		 * @param array $new_instance The new options.
		 * @param array $old_instance The previous options.
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = array();

			$instance['title']       = sanitize_text_field( $new_instance['title'] );
			$instance['subtitle']    = sanitize_text_field( $new_instance['subtitle'] );
			$instance['label']       = sanitize_text_field( $new_instance['label'] );
			$instance['price']       = sanitize_text_field( $new_instance['price'] );
			$instance['cta_text']    = sanitize_text_field( $new_instance['cta_text'] );
			$instance['cta_url']     = esc_url_raw( $new_instance['cta_url'] );
			$instance['new_tab']     = ! empty( $new_instance['new_tab'] ) ? sanitize_key( $new_instance['new_tab'] ) : '';
			$instance['is_featured'] = ! empty( $new_instance['is_featured'] ) ? sanitize_key( $new_instance['is_featured'] ) : '';

			foreach ( $new_instance['items'] as $key => $item ) {
				$instance['items'][ $key ]['id']   = sanitize_key( $item['id'] );
				$instance['items'][ $key ]['icon'] = sanitize_text_field( $item['icon'] );
				$instance['items'][ $key ]['text'] = sanitize_text_field( $item['text'] );
			}

			// Sort items by ids, because order might have changed.
			usort( $instance['items'], function ( $a, $b ) {
				return $a['id'] - $b['id'];
			} );

			return $instance;
		}

		/**
		 * Back-end widget form.
		 *
		 * @param array $instance The widget options.
		 */
		public function form( $instance ) {
			$title       = empty( $instance['title'] ) ? '' : $instance['title'];
			$subtitle    = empty( $instance['subtitle'] ) ? '' : $instance['subtitle'];
			$label       = empty( $instance['label'] ) ? '' : $instance['label'];
			$price       = empty( $instance['price'] ) ? '' : $instance['price'];
			$cta_text    = empty( $instance['cta_text'] ) ? '' : $instance['cta_text'];
			$cta_url     = empty( $instance['cta_url'] ) ? '' : $instance['cta_url'];
			$new_tab     = empty( $instance['new_tab'] ) ? '' : $instance['new_tab'];
			$is_featured = empty( $instance['is_featured'] ) ? '' : $instance['is_featured'];

			if ( ! isset( $instance['items'] ) ) {
				$instance['items'] = array(
					array(
						'id'          => 1,
						'icon'        => 'fa-check',
						'text'        => '',
					),
				);
			}

			// Page Builder fix when using repeating fields.
			if ( 'temp' === $this->id ) {
				$this->current_widget_id = $this->number;
			}
			else {
				$this->current_widget_id = $this->id;
			}
		?>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'subtitle' ) ); ?>"><?php esc_html_e( 'Subtitle:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'subtitle' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'subtitle' ) ); ?>" type="text" value="<?php echo esc_attr( $subtitle ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'label' ) ); ?>"><?php esc_html_e( 'Label:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'label' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'label' ) ); ?>" type="text" value="<?php echo esc_attr( $label ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'price' ) ); ?>"><?php esc_html_e( 'Price:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'price' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'price' ) ); ?>" type="text" value="<?php echo esc_attr( $price ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'cta_text' ) ); ?>"><?php esc_html_e( 'Button text:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'cta_text' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'cta_text' ) ); ?>" type="text" value="<?php echo esc_attr( $cta_text ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'cta_url' ) ); ?>"><?php esc_html_e( 'Button URL:', 'consultpress-pt' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'cta_url' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'cta_url' ) ); ?>" type="text" value="<?php echo esc_attr( $cta_url ); ?>" />
			</p>

			<p>
				<input class="checkbox" type="checkbox" <?php checked( $new_tab, 'on' ); ?> id="<?php echo esc_attr( $this->get_field_id( 'new_tab' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'new_tab' ) ); ?>" value="on" />
				<label for="<?php echo esc_attr( $this->get_field_id( 'new_tab' ) ); ?>"><?php esc_html_e( 'Open URL in a new tab!', 'consultpress-pt' ); ?></label>
			</p>

			<p>
				<input class="checkbox" type="checkbox" <?php checked( $is_featured, 'on' ); ?> id="<?php echo esc_attr( $this->get_field_id( 'is_featured' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'is_featured' ) ); ?>" value="on" />
				<label for="<?php echo esc_attr( $this->get_field_id( 'is_featured' ) ); ?>"><?php esc_html_e( 'Set this pricing package as featured!', 'consultpress-pt' ); ?></label>
			</p>

			<hr>

			<h4><?php esc_html_e( 'Details', 'consultpress-pt' ); ?></h4>

			<script type="text/template" id="js-pt-pricing-package-item-<?php echo esc_attr( $this->current_widget_id ); ?>">
				<div class="pt-sortable-setting  ui-widget  ui-widget-content  ui-helper-clearfix  ui-corner-all">
					<div class="pt-sortable-setting__header  ui-widget-header  ui-corner-all">
						<span class="dashicons  dashicons-sort"></span>
						<span><?php esc_html_e( 'Item', 'consultpress-pt' ); ?> - </span>
						<span class="pt-sortable-setting__header-title">{{text}}</span>
						<span class="pt-sortable-setting__toggle  dashicons  dashicons-minus"></span>
					</div>
					<div class="pt-sortable-setting__content">
						<p>
							<label for="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-icon"><?php esc_html_e( 'Icon:', 'consultpress-pt' ); ?></label> <br />
							<small><?php printf( esc_html__( 'Click on the icon below or manually input icon class from the %s website.', 'consultpress-pt' ), '<a href="http://fontawesome.io/icons/" target="_blank">FontAwesome</a>' ); ?></small>
							<input id="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-icon" name="<?php echo esc_attr( $this->get_field_name( 'items' ) ); ?>[{{id}}][icon]" type="text" value="{{icon}}" class="widefat  js-icon-input" /> <br><br>
							<?php foreach ( $this->font_awesome_icons_list as $icon ) : ?>
								<a class="js-selectable-icon  icon-widget" href="#" data-iconname="<?php echo esc_attr( $icon ); ?>"><i class="fa fa-lg <?php echo esc_attr( $icon ); ?>"></i></a>
							<?php endforeach; ?>
						</p>

						<p>
							<label for="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-text"><?php esc_html_e( 'Text:', 'consultpress-pt' ); ?></label>
							<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'items' ) ); ?>-{{id}}-text" name="<?php echo esc_attr( $this->get_field_name( 'items' ) ); ?>[{{id}}][text]" type="text" value="{{text}}" />
						</p>

						<p>
							<input name="<?php echo esc_attr( $this->get_field_name( 'items' ) ); ?>[{{id}}][id]" class="js-pt-pricing-package-id" type="hidden" value="{{id}}" />
							<a href="#" class="pt-remove-pricing-package-item  js-pt-remove-pricing-package-item"><span class="dashicons dashicons-dismiss"></span> <?php esc_html_e( 'Remove item', 'consultpress-pt' ); ?></a>
						</p>
					</div>
				</div>
			</script>

			<div class="pt-widget-pricing-package" id="pricing-package-items-<?php echo esc_attr( $this->current_widget_id ); ?>">
				<div class="pricing-package-items  js-pt-sortable-pricing-package-items"></div>
				<p>
					<a href="#" class="button  js-pt-add-pricing-package-item"><?php esc_html_e( 'Add new item', 'consultpress-pt' ); ?></a>
				</p>
			</div>

			<script type="text/javascript">
				(function( $ ) {
					// Repopulate the form.
					var pricingPackageJSON = <?php echo wp_json_encode( $instance['items'] ) ?>;

					// Get the right widget id and remove the added < > characters at the start and at the end.
					var widgetId = '<<?php echo esc_js( $this->current_widget_id ); ?>>'.slice( 1, -1 );

					if ( _.isFunction( ConsultPress.Utils.repopulatePricingPackageItems ) ) {
						ConsultPress.Utils.repopulatePricingPackageItems( pricingPackageJSON, widgetId );
					}

					// Make settings sortable.
					$( '.js-pt-sortable-pricing-package-items' ).sortable({
						items: '.pt-widget-single-pricing-package-item',
						handle: '.pt-sortable-setting__header',
						cancel: '.pt-sortable-setting__toggle',
						placeholder: 'pt-sortable-setting__placeholder',
						stop: function( event, ui ) {
							$( this ).find( '.js-pt-pricing-package-id' ).each( function( index ) {
								$( this ).val( index );
							});
						}
					});
				})( jQuery );
			</script>

		<?php
		}
	}
	register_widget( 'PW_Pricing_Package' );
}
